% This m-file is for single emitter localization using a vectorial
% dipole PSF with a Gaussian PSF model.

close all
clear all

model = 'standard2D';
% model = 'astigmatic3D';
% model = 'diffractivecolor2D';
% model = 'diffractivecolor3D';
switch model
    case 'standard2D'
        numparams = 5;
    case 'astigmatic3D'
        numparams = 6;
    case 'diffractivecolor2D'
        numparams = 6;
    case 'diffractivecolor3D'
        numparams = 7;
end
recalc2d = 1;
storeresults = 0;

ROI = 7;

load_str = 'bead45nm_8_31x31x21_teritary_spot1';
load(load_str)

[Nxpixels,Nypixels,~] = size(allspots);

allspots = allspots((ceil(Nxpixels/2)-floor(ROI/2):ceil(Nxpixels/2)+floor(ROI/2)),(ceil(Nypixels/2)-floor(ROI/2):ceil(Nypixels/2)+floor(ROI/2)),:,:);

[Nxpixels,Nypixels,Nzpixels] = size(allspots);
pixelsize = 80*1.49/552;
Ncfg = 1;
% 
Duxstr = 0;%parameters.yemit/parameters.pixelsize;
Duystr = 0;%parameters.xemit/parameters.pixelsize;
%%
% pupil and image size and coordinate sampling
ImageSizex = Nxpixels*pixelsize/2;
ImageSizey = Nypixels*pixelsize/2;
ximag = -ImageSizex+pixelsize/2:pixelsize:ImageSizex;
yimag = -ImageSizey+pixelsize/2:pixelsize:ImageSizey;
[XImage,YImage] = meshgrid(ximag,yimag);

% readout noise settings, and the variance value varfit used in the spot
% fitting, a value varfit >> 1 is the LS-limit, a value = 0 is the
% shot noise MLE-limit.
readnoisemean = 0.0;
readnoisestd = 0.0;
readnoisevariance = readnoisestd^2;
varfit = readnoisevariance;

%% Calculation of the PSF.

% optical parameter settings: polar and azimuthal angles, NA, refractive
% index medium, cover slip, immersion medium, wavelength (nm), magnification.
settings.model = model;
settings.pola = 0;
settings.azim = 0.0;
settings.NA = 1.49;
settings.refmed = 1.518;
settings.refcov = 1.518;
settings.refimm = 1.518;
settings.lambda = 552;
settings.pixelsize = pixelsize;
settings.Nxpixels = Nxpixels;
settings.Nypixels = Nypixels;

switch model
    case 'standard2D'
        phasedepth = 0.0;
        pitch = 0.75;
        scurvelength = 0;
        Fmax = 0.0;
    case 'astigmatic3D'
        phasedepth = 0.0;
        pitch = 0.75;
        scurvelength = 500;
        Fmax = 0.90;
    case 'diffractivecolor2D'
        phasedepth = settings.lambda/3;
        pitch = 0.75;
        scurvelength = 0;
        Fmax = 0.0;
    case 'diffractivecolor3D'
        phasedepth = settings.lambda/3;
        pitch = 0.50;
        scurvelength = 300;
        Fmax = 0.60;
end
settings.scurvelength = scurvelength;
settings.phasedepth = phasedepth;
settings.pitch = pitch;

%%

Nsteps = 1;

% allparsmean = zeros(Nsteps,numparams);
% allparsstd = zeros(Nsteps,numparams);
% allCRLBmean = zeros(Nsteps,numparams,numparams);
% allspots = zeros(Nypixels,Nxpixels,Ncfg);
switch model
    case {'astigmatic3D','diffractivecolor3D'}
        allzestmean = zeros(Nsteps,1);
        allzeststd = zeros(Nsteps,1);
        allCRLBzmean = zeros(Nsteps,1);
end

% parameter settings
paramaters.numparams = numparams;
paramaters.model = model;
paramaters.Nxpixels = Nxpixels;
paramaters.Nypixels = Nypixels;
paramaters.pixelsize = pixelsize;
paramaters.Nitermax = 1001;
paramaters.varfit = varfit;
paramaters.convpostolam = settings.pitch*settings.lambda;
paramaters.phasedepth = phasedepth;
paramaters.Fmax = Fmax;

% estimation of initial values for the fit parameters
tic
thetastore = initialvalues(paramaters,allspots);
toc

% Use initial fits as starting point in MLE.
tic
if recalc2d
    % MLE fit routine.
    [thetastore, sumdthetastore, meritstore, stdstore] = localization(paramaters,thetastore,allspots);
    % calculation of log-likelihood offset.
    meritoffset = meritoffsetcalc(allspots,varfit);
    for jiter = 1:paramaters.Nitermax
        meritstore(:,jiter) = meritstore(:,jiter)+meritoffset;
    end
end
toc

% calculation of Fisher-matrix and CRLB at found parameter-values
CRLBstore = FisherCRLB(paramaters,thetastore,readnoisevariance);

% calibration of the axial coordinate
switch model
    case 'astigmatic3D'
        ztrue = Duzstr*settings.lambda;
        Fest = thetastore(:,6);
        Fest = min(Fest,Fmax);
        Fest = max(Fest,-Fmax);
        sigmaz = thetastore(:,3);
        [Fmax_fit,zmax_fit,zest,sigma0] = zcalibrate(Fest,sigmaz,ztrue);
        scurvelength_fit = 2*Fmax_fit*zmax_fit;
        dof_fit = zmax_fit*sqrt(1-Fmax_fit^2);
        dzdF = zmax_fit*Fmax./sqrt(Fmax^2-Fest.^2)./(Fmax+sqrt(Fmax^2-Fest.^2));
        CRLBzstore = dzdF.^2.*squeeze(CRLBstore(6,6,:));
    case 'diffractivecolor3D'
        ztrue = Duzstr*settings.lambda;
        Fest = thetastore(:,7);
        Fest = min(Fest,Fmax);
        Fest = max(Fest,-Fmax);
        sigmaz = thetastore(:,3);
        [Fmax_fit,zmax_fit,zest,sigma0] = zcalibrate(Fest,sigmaz,ztrue);
        scurvelength_fit = 2*Fmax_fit*zmax_fit;
        dof_fit = zmax_fit*sqrt(1-Fmax_fit^2);
        zmax = 2*Fmax/(mean(Fest.*ztrue)/mean(ztrue.^2));
        dzdF = zmax*Fmax./sqrt(Fmax^2-Fest.^2)./(Fmax+sqrt(Fmax^2-Fest.^2));
        CRLBzstore = dzdF.^2.*squeeze(CRLBstore(7,7,:));
end
%%
% Analysis of the data.
%

%     outlier removal based on statistics of maximum log-likelihood, on spot
%     size, and on found position
outliers = (abs(thetastore(:,1))>2*pixelsize)|...
    (abs(thetastore(:,2))>2*pixelsize);

switch model
    case 'standard2D'
        outliers = outliers|(thetastore(:,3)>2*pixelsize);
    case {'astigmatic3D','diffractivecolor3D'}
        outliers = outliers|(thetastore(:,3)>3*pixelsize);
    case 'diffractivecolor2D'
        outliers = outliers|(thetastore(:,3)>2*pixelsize);
end

outliers = 0;

if (recalc2d)
    meritfinal = squeeze(meritstore(:,end));
    meanmeritfinal = mean(meritfinal);
    stdmeritfinal = std(meritfinal);
    outliers = outliers|(meritfinal<meanmeritfinal-3*stdmeritfinal);
end

% calculation of the mean and standard deviation of the results.
thetastore(:,1) = thetastore(:,1)-Duxstr;
thetastore(:,2) = thetastore(:,2)-Duystr;
switch model
    case {'astigmatic3D','diffractivecolor3D'}
        zest = zest-ztrue;
        zestmean = mean(zest(~outliers));
        zeststd = sqrt(var(zest(~outliers)));
        sigma0mean = mean(sigma0(~outliers));
        sigma0std = sqrt(var(sigma0(~outliers)));
end
thetamean = mean(thetastore(~outliers,:),1);
thetavar = var(thetastore(~outliers,:),1);
CRLBmean = mean(CRLBstore(:,:,~outliers),3);
thetastd = sqrt(thetavar);

% store mean and std in array for photon count runs
allparsmean = thetamean;
allparsstd = thetastd;
allCRLBmean = CRLBmean;
switch model
    case {'astigmatic3D','diffractivecolor3D'}
        allzestmean = zestmean;
        allzeststd = zeststd;
        allCRLBzmean = mean(CRLBzstore(~outliers));
end

%% Plot results
if (Nsteps==1)
        
    indexy = round(parameters.Mz/2);
    
    avspot = allspots(:,:,indexy);
    theta = thetastore(indexy,:);
    theta(1) = theta(1); %+Duxstr(indexy);
    theta(2) = theta(2); %+Duystr(indexy);
    [GaussPSFfit,dmudtheta] = poissonrate(paramaters,theta);
    
    figure;
    clims = [0, max(max(avspot))];
    imagesc(avspot,clims);
    colormap hot;
    axis off;
    axis square
    hold on;
    
    figure;
    clims = [0, max(max(avspot))];
    imagesc(GaussPSFfit,clims);
    colormap hot;
    axis off;
    axis square
    hold on;
    
    scrsz = get(0,'ScreenSize');
    figure('Position',[1*scrsz(4)/16 1*scrsz(4)/16 15*scrsz(3)/16 3*scrsz(4)/8]);
    
    subplot(1,3,1);
    hold on;
    plot(ximag,avspot((Nypixels+1)/2,:),'b','LineWidth',2);
    plot(yimag,avspot(:,(Nxpixels+1)/2),'r','LineWidth',2);
    plot(ximag,GaussPSFfit((Nypixels+1)/2,:),'--b','LineWidth',2);
    plot(yimag,GaussPSFfit(:,(Nxpixels+1)/2),'--r','LineWidth',2);
    xlabel('Position [\lambda/NA]','FontSize',12);
    ylabel('Normalized Intensity','FontSize',12);
    legend('Ix along x','Ix along y','fit along x','fit along y','Location','EastOutside');
    
    subplot(1,3,2)
    contour(XImage,YImage,avspot);
    title('spot','FontSize',12);
    xlabel('X [\lambda/NA]','FontSize',10);
    ylabel('Y [\lambda/NA]','FontSize',10);
    %colorbar;
    
    subplot(1,3,3)
    contour(XImage,YImage,GaussPSFfit);
    title('Gauss fit','FontSize',12);
    xlabel('X [\lambda/NA]','FontSize',10);
    ylabel('Y [\lambda/NA]','FontSize',10);
   
end

% save data
savename = [load_str '_gaussianfit_' num2str(ROI) 'x' num2str(ROI)];
if storeresults
    save(strcat(savename,'.mat'),'allPSFs','allspots','thetastore','meritstore','parameters');
end










